from datetime import date
import random
from django.http import Http404
from django.shortcuts import get_object_or_404, redirect, render
from django.views.generic.edit import CreateView, UpdateView, DeleteView
from django.views.generic import ListView, View, TemplateView, DetailView
from django.contrib.auth.mixins import LoginRequiredMixin
from django.urls import reverse_lazy
from club.models import Club
from core.models import Season
from core.utils import get_registration_year, season_getter
from core.views import ClubRequiredMixin
from .models import Player
from .forms import PlayerForm, PlayerUpdateForm
from django.contrib.auth.models import User
from django.core.exceptions import ObjectDoesNotExist
from django.contrib import messages
from django.db import IntegrityError


class PlayerCreateView(LoginRequiredMixin, ClubRequiredMixin, View):
    template_name = "players/player_form.html"

    def get(self, request):
        form = PlayerForm()
        return render(request, self.template_name, {"form": form})

    def post(self, request):
        form = PlayerForm(request.POST, request.FILES)

        if form.is_valid():
            player = form.save(commit=False)

            player.reg_year = get_registration_year()
            player.season = season_getter()

            # Generate a unique player_id
            for _ in range(10):
                candidate_id = str(random.randint(100, 999))
                if not Player.objects.filter(player_id=candidate_id).exists():
                    player.player_id = candidate_id
                    break
            else:
                form.add_error(None, "Unable to generate a unique player ID.")
                return self.form_invalid(form)

            # Require parental consent if minor
            if player.is_minor and not request.FILES.get("parental_consent"):
                messages.error(
                    request, "Parental consent document is required for minors.")
                return render(request, self.template_name, {"form": form, "show_consent": True})

            player.parental_consent = request.FILES.get("parental_consent")

            try:
                player.club = Club.objects.get(user=self.request.user)
            except ObjectDoesNotExist:
                form.add_error(None, "No club is associated with this user.")
                return self.form_invalid(form)

            try:
                player.save()
                messages.success(request, "Player registered successfully.")
                return redirect("club_players_list")
            except IntegrityError:
                form.add_error(
                    None, "Player ID already exists. Please try again.")
                return self.form_invalid(form)

        return render(request, self.template_name, {"form": form, })


class PlayerUpdateView(LoginRequiredMixin, ClubRequiredMixin, UpdateView):
    model = Player
    form_class = PlayerUpdateForm
    template_name = "players/player_update_form.html"
    context_object_name = "player"

    def form_valid(self, form):
        player = form.save(commit=False)

        if player.is_minor and not (self.request.FILES.get("parental_consent") or player.parental_consent):
            messages.error(self.request, "Parental consent is required for minors.")
            return self.form_invalid(form)


        if self.request.FILES.get("parental_consent"):
            player.parental_consent = self.request.FILES["parental_consent"]
            
            
        if player.status == "Rejected":
            player.status = "Unapproved"
            player.rejection_message = ""

        player.save()
        messages.success(self.request, "Player updated successfully.")
        return redirect(f"{reverse_lazy('club_players_list')}?status=Unapproved")
    
       
    def get_context_data(self, **kwargs):
        context = super().get_context_data(**kwargs)
        player = self.get_object()
        context["show_consent"] = player.is_minor
        return context


class PlayerDeleteView(LoginRequiredMixin, ClubRequiredMixin, DeleteView):
    model = Player
    template_name = "players/player_confirm_delete.html"
    success_url = reverse_lazy("club_players_list")

    def delete(self, request, *args, **kwargs):
        messages.success(request, "Player deleted successfully.")
        return super().delete(request, *args, **kwargs)


class GetPlayerDetailView(LoginRequiredMixin, ClubRequiredMixin, DetailView):
    model = Player
    template_name = "players/single_player.html"

    def get_queryset(self):
        club_id = get_object_or_404(Club, user=self.request.user)
        player_id = self.kwargs.get("pk")
        active_season = season_getter()

        return Player.objects.filter(
            pk=player_id,
            club_id=club_id,
            season=active_season,
            reg_year=get_registration_year(),
        )


class UpdatePlayerRecord(LoginRequiredMixin, ClubRequiredMixin, View):
    form_class = PlayerForm
    template_name = "players/player_update_form.html"
    success_url = reverse_lazy("player-list")

    def get(self, request, pk):
        player = get_object_or_404(Player, pk=pk)
        form = self.form_class(instance=player)
        return render(request, self.template_name, {'form': form, 'player': player})

    def post(self, request, pk):
        player = get_object_or_404(Player, pk=pk)
        form = self.form_class(request.POST, request.FILES, instance=player)
        if form.is_valid():
            form.save()
            return redirect(self.success_url)
        return render(request, self.template_name, {'form': form, 'player': player})


class ListClubPlayersByStatus(LoginRequiredMixin, ClubRequiredMixin, ListView):
    model = Player
    template_name = "players/club_players_by_status.html"
    context_object_name = "players"
    expected_status = ["Unapproved", "Approved", "Rejected"]

    def get_queryset(self):
        club = get_object_or_404(Club, user=self.request.user)
        status_filter = self.request.GET.get("status")

        # Only 404 if status is explicitly provided and invalid
        if status_filter and status_filter not in self.expected_status:
            raise Http404("Invalid status filter.")

        queryset = Player.objects.filter(
            club=club,
            season=season_getter(),
            reg_year=get_registration_year()
        )

        if status_filter:
            queryset = queryset.filter(status=status_filter)

        return queryset

    

    def get_context_data(self, **kwargs):
        context = super().get_context_data(**kwargs)
        status = self.request.GET.get("status")
        context['status_filter'] = status
        return context
